#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'Logger'

require 'aprs4r/APRSCall'
require 'aprs4r/APRSMessage'
require 'aprs4r/KISSDevice'


class AX25Socket

  attr_reader :name, :type, :enable, :duplicatePeriod

  @@logger = Logger::get_logger( "AX25Socket");


  def initialize( configuration)
    @@logger.info( "initialize( configuration)")

    @name = configuration.name
    @type = configuration.type
    @enable = configuration.enable
    @device = configuration.device
    @mode = configuration.mode
    @call = configuration.call
    @speed = configuration.speed
    @duplicatePeriod = configuration.duplicatePeriod
    @timeout = configuration.timeout
    
    if @enable 
      initAX25Socket()
    end

  end

  
  def initAX25Socket()
    @@logger.info( "initAX25Socket")

    if @mode.index( "kiss") == 0 
      @socket = KISSDevice.new( @device, @call, @speed, @timeout, @mode)
    end

    return
  end

  
  def readAPRSMessage()
    @@logger.info( "readAPRSMessage")

    message = APRSMessage.new

    frame = @socket.readFrame()

    if frame.nil?
      @@logger.warn( "empty KISS frame received")
      return nil
    end

    index = 0
    length = frame.length

    # read destination callsign 
    if length < index+7
      @@logger.warn( "no destination callsign")
      return nil
    end

    data = frame[index...index+7]
    destination = readCall( data, false)
    message.destination = destination
    @@logger.debug( "destination: #{destination}")
    index += 7 
    
    # read source callsign
    if length < index+7
      @@logger.warn( "no source callsign")
      return nil
    end

    data = frame[index...index+7]
    source = readCall( data, false)
    message.source = source
    @@logger.debug( "source: #{source}")
    index += 7 
    
    # read repeater path
    message.path = Array.new
    
    while (frame[index-1] & 0x01) == 0
      if length < index+7
        @@logger.warn( "no path callsigns")
        return nil
      end
      
      data = frame[index...index+7]
      entry = readCall( data)
      
      message.path << entry
      index += 7
    end
    
    # read protocol fields
    if length < index+2
      @@logger.warn( "no protocol fields")
      return nil
    end

    # read control field
    control = frame[index]
    @@logger.debug( "control: #{control.to_i}")
      
    if ((control & 0xE0) == 0) && ((control & 0x03) == 0x03)
      @@logger.debug( "type: UI-Frame")
    end
    
    # read pid field
    pid = frame[index+1]
    @@logger.debug( "pid: #{pid.to_i}")
    
    if pid == 0xF0 
      @@logger.debug( "no layer 3")
    end

    index += 2
      
    
    # read payload
    if length < index
      @@logger.warn( "no payload")
      return nil
    end
    
    payload = frame[index...frame.length]
    message.payload = payload.chomp if payload
    
    return message
  end


  def writeAPRSMessage( message)
    @@logger.info( "writeAPRSMessage( message)")
    
    @@logger.debug( "message: #{message.to_s}")

    data = String.new

    # write destination call
    data << writeCall( message.destination, 0x60)

    # write source call
    path = message.path

    if path.length == 0 
      data << writeCall( message.source, 0xE1)
    else 
      data << writeCall( message.source, 0xE0)
    end
    
    # write path 
    for i in 0...path.length
      call = path[i]

      if i != path.length - 1
        data << writeCall( call, 0x60)
      else 
        data << writeCall( call, 0x61)
      end
    end

    # write control field
    data << ((0xE0 & 0x00) | 0x03)

    # write pid field
    data << 0xF0

    # write payload
    data << message.payload


    @socket.writeFrame( data)
    
    return data
  end


  def readCall( data, pathcalls = true)
    @@logger.info( "readCall( data)")

    call = APRSCall.new

    # parse name
    name = String.new
    for i in 0..5
      value = (data[i] >> 1).chr

      if value != ' '
        name << value
      end
    end
    call.call = name

    # parse ssid
    ssid = (data[6] & 0x1E) >> 1
    call.ssid = ssid

    # FIXME: parse H-bit
    repeated = (data[6] & 0x80)
    if repeated == 0x80 && pathcalls == true
      call.repeated = true
    end

    return call.to_s
  end


  def writeCall( callsign, flags)
    @@logger.info( "writeCall( call, flags)")
    
    data = String.new
    
    call = APRSCall.parse( callsign)

    # write name
    name = call.call
    @@logger.debug( "call: #{call.call}")
    for i in 0...6
      if i < name.length
        data << (name[i] << 1)
      else 
        blank = " "
        data << (blank[0] << 1)
      end
    end

    # write ssid
    @@logger.debug( "ssid: #{call.ssid}")
    repeated = 0 
    if call.repeated? 
      repeated = 0x80
    end
    ssid = (repeated | flags | (call.ssid.to_i << 1).to_i)

    data << ssid

    return data
  end

end
