#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'Logger'

require 'aprs4r/APRSCall'


class APRSMessage

  @@APRS_PATH_RELAY = "RELAY"
  @@APRS_PATH_WIDE  = "WIDE"
  @@APRS_PATH_TRACE = "TRACE"

  @@APRS_POSITION = { '!' => true, '=' => true, '/' => true, '@' => true}
  @@APRS_POSITION_LATITUDE_INDEX = { '!' => 1, '=' => 1, '/' => 8, '@' => 8}
  @@APRS_POSITION_LONGITUDE_INDEX = { '!' => 10, '=' => 10, '/' => 17, '@' => 17}

  @@APRS_SYMBOL_TABLE_INDEX = { '!' => 9, '=' => 9, '/' => 16, '@' => 16 }
  @@APRS_SYMBOL_CODE_INDEX = { '!' => 19, '=' => 19, '/' => 26, '@' => 26 }

  @@APRS_SYMBOL_CODE_NAME = { 
    "#" => "digi", # digipeater 
    "&" => "gateway", # gateway (internet gateway)
    "_" => "weather", # weather station 
    "-" => "house", # house
    "Y" => "yacht", # yacht
    "s" => "boat", # boat
    ">" => "car", # car
  }
    
    
  attr_reader :source, :destination, :path, :payload
  attr_writer :source, :destination, :path, :payload


  @@logger = Logger.get_logger( 'APRSMessage')


  def initalize( source = APRSCall.new, destination = APRSCall.new, path = Array.new, payload = nil)
    @@logger.info( "initialize( source, destination, path, payload)")
    
    @source = source
    @destination = destination
    @path = path
    @payload = payload

    @type = nil
    
    return
  end


  def clone
    @@logger.info( "clone")
    
    message = APRSMessage.new

    message.source = @source.clone
    message.destination = @destination.clone

    path = Array.new
    @path.each{ |entry| path << entry.clone} if @path
    message.path = path

    message.payload = String.new( @payload)

    return message
  end


  def duplicate_key
    @@logger.info( "duplicate_key")
    
    key = String.new

    key << source.to_s << ">" << destination.to_s
    key << ":" << payload

    return key
  end


  def to_yaml_properties
    %w{@source @destination @path @payload}
  end


  def type
    @@logger.info( "type")

    if payload.length > 0 
      type = payload[0].chr
    end

    return type
  end


  def has_symbol?
    @@logger.info( "has_symbol?")

    if payload.length > 0 
      type = payload[0].chr
      return @@APRS_SYMBOL_TABLE_INDEX.has_key?( type)
    end
    
    return false
  end


  def symbol_table
    @@logger.info( "symbol_table")

    if payload.length > 0 
     type = payload[0].chr

      index = @@APRS_SYMBOL_TABLE_INDEX[type]

      if index < payload.length
        return payload[index].chr
      end
    end

    return nil
  end


  def symbol_code
    @@logger.info( "symbol_code")

    if payload.length > 0 
      type = payload[0].chr

      index = @@APRS_SYMBOL_CODE_INDEX[type]

      if index < payload.length
        return payload[index].chr
      end
    end
    
    return nil
  end

  
  def symbol_name
    @@logger.info( "symbol_name")

    code = symbol_code

    @@logger.debug( "code: #{code}")

    name = @@APRS_SYMBOL_CODE_NAME[code]

    @@logger.debug( "name: #{name}")

    return name
  end


  def has_position?
    @@logger.info( "has_position?")

    if payload.length > 0 
      type = payload[0].chr
      return @@APRS_POSITION.has_key?( type)
    end

    return false
  end
  

  def latitude
    @@logger.info( "latitude")

    latitude = 0.0

    if has_position?
      type = payload[0].chr

      index = @@APRS_POSITION_LATITUDE_INDEX[type]
      length = 8

      @@logger.debug( "latitude: #{payload[index...index+length]}")
      
      if index + length < payload.length
        degree = payload[index...index+2].to_f
        minute = payload[index+2...index+4].to_f
        second = payload[index+5...index+7].to_f

        latitude = degree + ((minute + (second / 60.0)) / 60.0)

        if payload[index+8] == 'S'
          latitude *= -1.0
        end
      end
    end

    return latitude
  end


  def longitude
    @@logger.info( "longitude")

    longitude = 0.0

    if has_position?
      type = payload[0].chr

      index = @@APRS_POSITION_LONGITUDE_INDEX[type]
      length = 9

      @@logger.debug( "longitude: #{payload[index...index+length]}")
      
      if index + length < payload.length
        degree = payload[index...index+3].to_f
        minute = payload[index+3...index+5].to_f
        second = payload[index+6...index+8].to_f

        longitude = degree + ((minute + (second / 60.0)) / 60.0)

        if payload[index+9] == 'W'
          longitude *= -1.0
        end
      end
    end

    return longitude
  end

  
  def is_message?
    @@logger.info( "is_message?( message)")

    type = false

    if payload.length > 0 
      type = payload[0].chr
    end

    return type == ':'
  end

  
  def is_query?
    @@logger.info( "is_query?( message)")

    type = false

    if payload.length > 0 
      type = payload[0].chr
    end

    return type == '?'
  end


  def to_s
    buffer = String.new

    # buffer << "APRSMessage: "
    buffer << @source.to_s << " -> "
    buffer << @destination.to_s << " via "
    
    path = String.new
    for i in 0...@path.length

      if i != 0 
        path = path + ", "
      end

      path = path + @path[i].to_s
    end
    path = path

    buffer << "[" << path << "]: "
    buffer << "(" << @payload << ")"

    return buffer.to_s
  end

  
  def APRSMessage.APRS_PATH_RELAY
    return @@APRS_PATH_RELAY
  end

  def APRSMessage.APRS_PATH_WIDE
    return @@APRS_PATH_WIDE
  end

  def APRSMessage.APRS_PATH_TRACE
    return @@APRS_PATH_TRACE
  end

end
