#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'serialport'
require 'thread'
require 'timeout'
require 'io/nonblock'

require 'Logger'


class KISSDevice

  @@FEND = 0xC0
  @@FESC = 0xDB
  @@TFEND = 0xDC
  @@TFESC = 0xDD

  @@logger = Logger.get_logger( "KISSDevice")
  

  def initialize( device, call, speed, timeout, mode)
    @@logger.info( "initialize( device, call, speed, timeout, mode)")
    
    @device = device
    @call = call
    @speed = speed
    @timeout = timeout
    @mode = mode

    @mutex = Mutex.new
    
    initKISSDevice()

    return
  end


  def initKISSDevice()
    @@logger.info( "initKISSDevice")

    @mutex.synchronize do

      if @port 
        @port.close
      end
      
      @port = SerialPort.new( @device, 9600, 8, 1, SerialPort::NONE)
      @port.nonblock = true 
      @port.sync = true
      
      @port.flow_control = SerialPort::SOFT

      if mode.index( "tnc2/northlink") != nil
        initTNC2Northlink
      elsif mode.index( "tnc2/tapr") != nil
        initTNC2TAPR
      elsif mode.index( "kenwood") != nil
        initKenwood
      else
        initDefault
      end
      
    end

    return
  end


  def reinitKISSDevice()
    @@logger.info( "reinitKISSDevice")

    initKISSDevice()

    return
  end


  def initTNC2Northlink
    @@logger.info( "initTNC2Northlink")

    # send exit frame 
    data = String.new
      
    data << @@FEND
    data << 0xFF
    data << @@FEND
    data << "\r" 

    @@logger.debug( "send KISS exit frame");
    @port.write( data)
    sleep 1

    data = String.new
    data << ?\C-X
    data << ?\C-[
    data << "@K"
    data << "\r"

    @@logger.debug( "send KISS init sequence")
    @port.write( data)
    sleep 1

    return
  end


  def initTNC2TAPR
    @@logger.info( "initTNC2TAPR")
    
    # send exit frame 
    data = String.new
      
    data << @@FEND
    data << 0xFF
    data << @@FEND
    data << "\r" 

    @@logger.debug( "send KISS exit frame");
    @port.write( data)
    sleep 1
      
    # enable kiss mode
    # @@logger.debug( "send: mycall #{@call}");
    # @port.write( "mycall #{@call}\r")
    # sleep 1

    # @@logger.debug( "send: hbaud #{@speed}");
    # @port.write( "hbaud #{@speed}\r");
    # sleep 1

    @@logger.debug( "send: kiss on ");
    @port.write( "kiss on\r");
    sleep 1

    @@logger.debug( "send: restart");
    @port.write( "restart\r")
    sleep 1

    return
  end


  def initKenwood
    @@logger.info( "initKenwood")

    @@logger.debug( "restart kenwood tnc")
    @port.write( "TC 1\r")
    sleep 2
    @port.write( "TC 0\r")
    sleep 2
      

    # enable kiss mode
    @@logger.debug( "send: mycall #{@call}");
    @port.write( "mycall #{@call}\r")
    sleep 1
    
    @@logger.debug( "send: hbaud #{@speed}");
    @port.write( "hbaud #{@speed}\r");
    sleep 1
    
    @@logger.debug( "send: kiss on ");
    @port.write( "kiss on\r");
    sleep 1
    
    @@logger.debug( "send: restart");
    @port.write( "restart\r")
    sleep 1
    
    return
  end


  def initDefault
    @@logger.info( "initDefault")
    
    # send exit frame 
    data = String.new
      
    data << @@FEND
    data << 0xFF
    data << @@FEND
    data << "\r" 

    @@logger.debug( "send KISS exit frame");
    @port.write( data)
    sleep 1
      
    # enable kiss mode
    # @@logger.debug( "send: mycall #{@call}");
    # @port.write( "mycall #{@call}\r")
    # sleep 1

    # @@logger.debug( "send: hbaud #{@speed}");
    # @port.write( "hbaud #{@speed}\r");
    # sleep 1

    @@logger.debug( "send: kiss on ");
    @port.write( "kiss on\r");
    sleep 1

    @@logger.debug( "send: restart");
    @port.write( "restart\r")
    sleep 1

    return
  end


  def readFrame
    @@logger.info( "readFrame")

    frame = nil

    begin 
      Timeout::timeout( @timeout) do 
        # read until FEND 
        value = @port.getc
        while value != @@FEND 
          
          if value.nil?
            return nil
          end 
          
          @@logger.warn( "read wire byte: #{value} #{value.chr}")
          
          value = @port.getc
        end
        
        # read all FENDs
        while value == @@FEND
          value = @port.getc
        end

        if value.nil? 
          return nil
        end
      

        # read command byte
        comand = value
        
        # read frame
        frame = String.new
        value = @port.getc
        while value != @@FEND
          frame << value
          value = @port.getc
          
          if value.nil?
            return nil
          end
        end
      end
    rescue Exception => ex
      @@logger.warn( "readFrame::ex: " + ex)
      reinitKISSDevice()
      return nil
    end

    return decodeKISS( frame)
  end


  def writeFrame( message)
    @@logger.info( "writeFrame( message)")

    data = String.new

    data << @@FEND
    data << 0x00
    # data << 0x7E
    data << encodeKISS( message)
    # data << 0x00
    # data << 0x00
    # data << 0x7E
    data << @@FEND

    @mutex.synchronize do
      @port.puts( data)
    end

    return
  end


  def encodeKISS( payload)
    @@logger.info( "encodeKISS( payload)")

    data = String.new

    for i in 0...payload.length

      value = payload[i]

      # do kiss encoding (FEND->FESC+TFEND, FESC->FESC+TFESC)
      if value == @@FEND
        data << @@FESC << @@TFEND
      elsif value == @@FESC
        data << @@FESC << @@TFESC
      else 
        data << value
      end

    end

    return data    
  end


  def decodeKISS( payload)
    @@logger.info( "decodeKISS( payload)")

    if payload.nil?
      @@logger.warn( "nil payload field")
      return nil
    end

    data = String.new

    for i in 0...payload.length

      value = payload[i]

      # do kiss decoding (FESC+TFEND->FEND, FESC+TFESC->FESC)
      if value == @@FESC
        value = payload[i+=1]
   
        if value == @@TFEND
          data << @@FEND
        elsif value == @@TFESC
          data << @@FESC
        end
      else 
        data << value
      end 

    end

    return data
  end
  
end
