#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'Logger'

require 'aprs4r/APRSMessage'
require 'aprs4r/Plugin'
require 'aprs4r/XMLDataPluginConfiguration'

class XMLDataPlugin < Plugin

  @@logger = Logger.get_logger( "XMLDataPlugin")

  
  def initialize( configuration)
    @@logger.info( "initialize( configuration)")

    super( configuration)

    @file = configuration.file
    @device = configuration.device

    @messages = Hash.new
    @timestamps = Hash.new
    
    return
  end

  
  def start 
    @@logger.info( "start")

    if ! @enable
      return
    end

    registerListener

    Thread.new { run() } 

    return
  end

  
  def registerListener
    @@logger.info( "registerListener")

    if @enable
      SocketManager.addListener( @device, self)
    end

    return
  end


  def unregisterListener
    @@logger.info( "registerListener")

    if @enable
      SocketManager.removeListener( @device, self)
    end

    return
  end


  def receiveAPRSMessage( message)
    @@logger.info( "receiveAPRSMessage( message)")

    if message.nil? 
      return
    end

    source = message.source.to_s
    now = Time.now

    # only process position reports
    if message.has_symbol? && message.symbol_name != "" && message.has_position?

      @@logger.debug( "table: #{message.symbol_table}")
      @@logger.debug( "code: #{message.symbol_code}")
      @@logger.debug( "name: #{message.symbol_name}")

      # remove old aprs message
      if @messages.has_key?( source)
        @messages.delete( source)
      end
      
      if @timestamps.has_key?( source)
        @timestamps.delete( source)
      end

      # insert new message
      @messages[source] = message
      @timestamps[source] = now
      
    else 
      @@logger.debug( "unknown message type: #{message.type}")
    end

    return
  end


  def run 
    @@logger.info( "run")

    while true 

      now = Time.now

      # clean up message hash
      @timestamps.each{ |key, timestamp|
        if now - timestamp > 1800
          @messages.delete( key)
          @timestamps.delete( key)
        end
      }

      file = File.new( @file, "w")

      file.puts( "<stations>")

      @messages.each{ |key, message|

        if message
          source = message.source
          destination = message.destination
          path = message.path

          path_list = String.new
          for i in 0...path.length
            if i != 0 
              path_list += ","
            end

            path_list += path[i].to_s
          end

          timestamp = @timestamps[source]

          # start station section
          name = "else"
          name = message.symbol_name if message.has_symbol? && message.symbol_name != ""
          station_open = sprintf( "<station source=\"%s\" destination=\"%s\" path=\"%s\" name=\"%s\">", source, destination, path_list, name)
          file.puts( station_open)

          # write position
          position = "<position lat=\"00.0000\" lng=\"000.0000\"/>"
          position = sprintf( "<position lat=\"%02.04f\" lng=\"%03.04f\"/>", message.latitude, message.longitude) if message.has_position?

          file.puts( position)
                                
          # write timestamp
          time = sprintf( "<time year=\"%04.4i\" month=\"%02.2i\" day=\"%02.2i\" hour=\"%02.2i\" minute=\"%02.2i\"/>", now.year, now.month, now.day, now.hour, now.min)
          file.puts( time)

          # write comment section
          file.puts( "<comment>")
          file.puts( "aprs comment")
          file.puts( "</comment>")

          # write status section
          file.puts( "<status>")
          file.puts( "aprs status")
          file.puts( "</status>")

          # write plain section
          # file.puts( "<plain>")
          # file.puts( message.to_s)
          # file.puts( "</plain>")

          # end station section
          file.puts( "</station>")
        end
      }

      file.puts( "</stations>")

      file.close

      sleep 10

    end

  end
end
