#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'socket'
require 'thread'
require 'timeout'
require 'io/nonblock'

require 'Logger'

require 'aprs4r/APRSCall'


class ISSocket

  attr_reader :name, :type, :enable, :duplicatePeriod

  @@logger = Logger.get_logger( "ISSocket")


  def ISSocket.generatePassword( value)
    @@logger.info( "generatePassword( value)")

    callsign = APRSCall.parse( value)
    call = callsign.call

    if call.nil?
      return 
    end

    # init value
    value = 0x73E2
    i = 0

    while i < call.length
      value = value ^ (call[i].to_i << 8)
      value = value ^ (call[i+1].to_i)

      i += 2
    end

    return (value & 0x7FFF).to_s
  end


  def initialize( configuration)
    @@logger.info( "initialize( configuration)")

    @name = configuration.name
    @type = configuration.type
    @enable = configuration.enable

    @host = configuration.host
    @port = configuration.port
    @username = configuration.username
    @password = ISSocket.generatePassword( @username)
    @filter = configuration.filter
    @timeout = configuration.timeout
    @duplicatePeriod = configuration.duplicatePeriod

    @mutex = Mutex.new

    if @enable 
      initISSocket()
    end
    
    return
  end
  

  def initISSocket()
    @@logger.info( "initISSocket")

    @socket = nil

    begin 
      @socket = TCPSocket.new( @host, @port)

      @socket.nonblock = true
      @socket.sync = true 
    
      # send login sequence
      username = "user " + @username
      password = "pass " + @password
      software = "vers aprs4r 0.8"
      
      @socket.puts( username + " " + password + " " + software)
      @socket.puts( @filter)
    rescue Exception => ex
      @@logger.warn( "initISSocket::ex: #{ex}")
    end

    return
  end


  def reinitISSocket()
    @@logger.info( "reinitISSocket")

    begin
      Timeout::timeout( 5) do
        if ! @socket.nil?
          @socket.close
        end
      end
    rescue Exception => ex
      @@logger.warn( "reinitISSocket::ex: #{ex}")
    end

    @socket = nil

    if @enable
      initISSocket()
    end

    return
  end


  def readAPRSMessage
    @@logger.info( "readAPRSMessage")

    message = APRSMessage.new

    data = readData

    if data.nil?
      return nil
    end

    headerIndex = data.index( ':')
    
    if !headerIndex.nil?
        
      # header
      header = data[0...headerIndex]
      
      sourceIndex = header.index( '>')
      
      if sourceIndex != nil 
        message.source = APRSCall.parse( header[0...sourceIndex])
      end

      # path
      message.path = Array.new
      path = header[sourceIndex+1...header.length].split( ',')
      message.destination = APRSCall.parse( path[0]) 

      path.delete( 0)
      
      path.each{ |entry|
        call = APRSCall.parse( entry)
        message.path << call
      }

      # body
      body = data[headerIndex+1...data.length]
      message.payload = body
    end
    
    return message
  end

  
  def writeAPRSMessage( message) 
    @@logger.info( "writeAPRSMessage( message)")
  
    if message.nil?
    	return
    end
    
    if message.source.nil? || message.destination.nil? 
      @@logger.warn( "NIL\n")
      return
    end

    data = String.new

    data = data + message.source.to_s + ">" + message.destination.to_s

    path = message.path
    
    for i in 0...path.length
      data = data + "," + path[i].to_s
    end

    data = data + ":" + message.payload

    writeData( data)
    
    return
  end


  def readData
    @@logger.info( "readData")

    @mutex.synchronize do 
      if @socket.nil?
        reinitISSocket()
      end
    end

    line = nil

    begin
      Timeout::timeout( @timeout) do

        if ! @socket.nil?
          line = @socket.gets
        end
        
        if !line.nil?
          line = line.chomp
        end

        @@logger.debug( "readData::line: -#{line}-")

      end
    rescue Exception => ex
      @@logger.warn( "readData::ex: " + ex)
      reinitISSocket()
      # FIXME 
      return nil
    end until line !~ /^#/
    
    return line
  end
  

  def writeData( data)
    @@logger.info( "writeData( data)")

    @mutex.synchronize do 
      if @socket.nil?
        reinitISSocket()
      end
    end

    @mutex.synchronize do
      begin
        if ! @socket.nil?
          @@logger.debug( "writeData::data: -#{data}-")
          @socket.puts( data)
          @socket.flush
        end
      rescue Exception => ex

        @@logger.warn( "ex: " + ex)
        reinitISSocket()
        
        if ! @socket.nil?
          @socket.puts( data)
        end
      end
    end

    return
  end
  
end
