#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'thread'
require 'io/nonblock'
require 'serialport'

require 'Logger'


class KISSDevice

  @@FEND = 0xC0
  @@FESC = 0xDB
  @@TFEND = 0xDC
  @@TFESC = 0xDD

  @@logger = Logger.get_logger( "KISSDevice")
  

  def initialize( device)
    @@logger.info( "initialize( device)")
    
    @device = device
    @mutex = Mutex.new
    
    initKISSDevice()

    return
  end


  def initKISSDevice()
    @@logger.info( "initKISSDevice")

    if @port 
      @port.close
    end

    @port = SerialPort.new( @device, 9600, 8, 1, SerialPort::NONE)
    @port.nonblock = true 
    @port.sync = true

    @port.flow_control = SerialPort::SOFT

    @mutex.synchronize do

      # send exit frame 
      data = String.new
      
      data << @@FEND
      data << 0xFF
      data << @@FEND
      data << "\r" 

      @@logger.debug( "send KISS exit frame");
      @port.write( data)
      sleep 2

      # enable kiss mode
      @@logger.debug( "send: mycall");
      @port.write( "mycall do5mc\r")
      sleep 2
      print( "line: #{@port.gets}\n")
      print( "line: #{@port.gets}\n")

      @@logger.debug( "send: hbaud 1200");
      @port.write( "hbaud 1200\r");
      sleep 2
      print( "line: #{@port.gets}\n")
      print( "line: #{@port.gets}\n")

      @@logger.debug( "send: kiss on ");
      @port.write( "kiss on\r");
      sleep 2
      print( "line: #{@port.gets}\n")
      print( "line: #{@port.gets}\n")

      @@logger.debug( "send: restart");
      @port.write( "restart\r")
      sleep 2
      print( "line: #{@port.gets}\n")
#      print( "line: #{@port.gets}\n")
    end

    @@logger.debug( "after config")

    return
  end


  def reinitKISSDevice()
    @@logger.info( "reinitKISSDevice")

    initKISSDevice()

    return
  end


  def readFrame
    @@logger.info( "readFrame")

    # read until FEND 
    value = @port.getc
    while value != @@FEND 

      if value.nil?
        return nil
      end 
      
      @@logger.warn( "read wire byte: #{value} #{value.chr}")

      value = @port.getc
    end

    # read all FENDs
    while value == @@FEND
      value = @port.getc
    end

    if value.nil? 
      return nil
    end
      

    # read command byte
    comand = value

    # read frame
    frame = String.new
    value = @port.getc
    while value != @@FEND
      frame << value
      value = @port.getc

      if value.nil?
        return nil
      end
    end

    return decodeKISS( frame)
  end


  def writeFrame( message)
    @@logger.info( "writeFrame( message)")

    data = String.new

    data << @@FEND
    data << 0x00
    # data << 0x7E
    data << encodeKISS( message)
    # data << 0x00
    # data << 0x00
    # data << 0x7E
    data << @@FEND

    @mutex.synchronize do
      @port.puts( data)
    end

    return
  end


  def encodeKISS( payload)
    @@logger.info( "encodeKISS( payload)")

    data = String.new

    for i in 0...payload.length

      value = payload[i]

      # do kiss encoding (FEND->FESC+TFEND, FESC->FESC+TFESC)
      if value == @@FEND
        data << @@FESC << @@TFEND
      elsif value == @@FESC
        data << @@FESC << @@TFESC
      else 
        data << value
      end

    end

    return data    
  end


  def decodeKISS( payload)
    @@logger.info( "decodeKISS( payload)")

    data = String.new

    for i in 0...payload.length

      value = payload[i]

      # do kiss decoding (FESC+TFEND->FEND, FESC+TFESC->FESC)
      if value == @@FESC
        value = payload[i+=1]
   
        if value == @@TFEND
          data << @@FEND
        elsif value == @@TFESC
          data << @@FESC
        end
      else 
        data << value
      end 

    end

    return data
  end
  
end
